/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: OCF waitForCard sample.
 *
 */

package ocfwait;
import opencard.core.service.CardRequest;
import opencard.core.service.CardServiceException;
import opencard.core.service.SmartCard;
import opencard.core.terminal.CardTerminalException;
import opencard.core.util.OpenCardPropertyLoadingException;
import opencard.opt.iso.fs.CardFilePath;
import opencard.opt.iso.fs.FileAccessCardService;
import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.LED;
import jp.co.ricoh.dsdk.panel.LEDExecutionException;
import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.LED.Operation;
import jp.co.ricoh.dsdk.panel.LED.Type;
import jp.co.ricoh.dsdk.panel.event.KeyAdapter;
import jp.co.ricoh.dsdk.panel.event.KeyEvent;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * OcfWait
 * This is a sample application using the waitForCard method of the OpenCard Framework.
 * 
 * OcfWait
 * OpenCard Frameworkで、waitForCardメソッドを利用するサンプルプログラムです。
 */
public class OcfWait implements Xlet {
	
	/**
	 * Indicates the file path that stores the serial number of the card.
	 * 
	 * シリアル番号が格納されたファイル
	 */
	private static final CardFilePath filePath = new CardFilePath(":3F00:0002");
	
	/**
	 * Indicates the size of the file that stores the serial number of the card.
	 * 
	 * シリアル番号が格納されたファイルのサイズ
	 */
	private static final int fileLen = 8;

	/**
	 * The thread to be used to wait for cards.
	 * 
	 * waitForCardを実行するスレッド
	 */
	private Thread thread;
	
	/**
	 * The root frame.
	 * 
	 * ルートフレーム
	 */
	private Frame frame;
	
	/**
	 * The label to be used to display a message.
	 * 
	 * メッセージ表示ラベル
	 */
	private Label msgLabel;
	
	/**
	 * The label to be used to display a serial number.
	 * 
	 * シリアル番号表示ラベル
	 */
	private Label serialLabel;
	
	/**
	 * This is an initialization process.
	 * 1.obtains the root frame.
	 * 2.creates GUI.
	 * 3.sets LED state.
	 *  
	 * 初期化処理です。
	 * 1.ルートフレームを取得する。
	 * 2.GUIを作成する。
	 * 3.LEDを設定する。
	 */
	synchronized public void initXlet(XletContext xletContext) throws XletStateChangeException {
		try {
			Container parent = xletContext.getContainer();
			while (!(parent instanceof Frame)) {
				parent = parent.getParent();
				if (parent == null)	break;
			}
			if (parent == null) {
				return;
			}
			frame = (Frame) parent;

			createGUI();
			
		} catch (UnavailableContainerException e) {
            throw new XletStateChangeException(e.getMessage());

        } catch (Exception e){
            throw new XletStateChangeException(e.getMessage());
        }
		
        try {
			SmartCard.start();
			
		} catch (OpenCardPropertyLoadingException e) {
			setMessage(new Text("ERR_PROPERTYLOAD"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (CardServiceException e) {
			setMessage(new Text("ERR_CARDSERVICE"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (CardTerminalException e) {
			setMessage(new Text("ERR_CARDTERMINAL"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (ClassNotFoundException e) {
			setMessage(new Text("ERR_CLASSNOTFOUND"));
			e.printStackTrace();
			throw new XletStateChangeException(e.getMessage());
		} catch (Throwable t) {
			setMessage(new Text("ERR_THROWABLE"));
			t.printStackTrace();
			throw new XletStateChangeException(t.getMessage());
		}
        
        setLED(Type.START, Operation.GRN_ON);
	}

	/**
	 * Activation
	 * 
	 * 活性化。
	 */
	synchronized public void startXlet() throws XletStateChangeException {
	}
	
	/**
	 * Stop.
	 * 
	 * 停止。
	 */
	synchronized public void pauseXlet() {
	}

	/**
	 * End process.
	 * This process is similar to the process followed after the clear/stop key is pressed.
	 * 
	 * 終了処理。
	 * クリアストップキー押された時と同様の処理を行う。
	 */
	synchronized public void destroyXlet(boolean destroy) throws XletStateChangeException {
		pushClearStopKey();
		
		try {
			SmartCard.shutdown();
			
		} catch (CardTerminalException e) {
			setMessage(new Text("ERR_CARDTERMINAL"));
			e.printStackTrace();
			
		}
	}
	
	/**
	 * GUI creation.
	 * This is called from the initXlet method, and creates GUI on the operation panel.
	 * 1.creates a title window.
	 * 2.creates a title label.
	 * 3.creates a message label.
	 * 4.creates a serial number label.
	 * 5.registers a KeyListener to the root frame for processing KeyEvents. 
	 * 
	 * GUI作成。
	 * initXletメソッドから呼び出され、オペパネにＧＵＩを作成します。
	 * 1.タイトル表示ウィンドウを作成する。
	 * 2.タイトル表示ラベルを作成する。
	 * 3.メッセージ表示ラベルを作成する。
	 * 4.シリアル番号表示ラベルを作成する。
	 * 5.キーリスナーを登録して、キーイベントを処理する。
	 */
	private void createGUI() {
		Window titleWindow = new Window(frame);
		titleWindow.setAttributeType(AttributeType.INVERSE);
		titleWindow.setLocation(0, 0);
		titleWindow.setSize(frame.getWidth(), 32);

		Label titleLabel = new Label(new Text("APP_TITLE"));  
		titleLabel.setFont(Font.F16);
		titleLabel.setAttributeType(AttributeType.INVERSE);
		titleLabel.setLocation(0, 0);
		titleLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(titleLabel);
		
		msgLabel = new Label(new Text("MSG_PUSHSTART"));  
		msgLabel.setFont(Font.F16);
		msgLabel.setAttributeType(AttributeType.INVERSE);
		msgLabel.setLocation(0, 16);
		msgLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(msgLabel);

		serialLabel = new Label();
		serialLabel.setSize(titleWindow.getWidth()-200, 16);
		serialLabel.setLocation(
				100 ,
				(frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 2);
		frame.add(serialLabel);
		setSerial(null);
		
		frame.addKeyListener(
				new KeyAdapter() {
					public void keyPressed(KeyEvent keyEvent) {
						switch (keyEvent.getKeyCode()) {
						case KeyEvent.KEY_START:     // Start key
							pushStartKey();
							break;
						case KeyEvent.KEY_CLR_STOP:  // Clear/Stop key
							pushClearStopKey();
							break;
						}
					}
				}
			);
	}
	
	/**
	 * The process followed after the start key is pressed.
	 * 1.creates and starts a thread which invokes the waitForCard method.
	 * 2.sets LED state.
	 * 
	 * スタートキーが、押された時の処理
	 * 1.waitForCardを実行するスレッドを開始する。
	 * 2.LEDを設定する。
	 */
	synchronized private void pushStartKey() {
		if ((thread==null) || !thread.isAlive()) {
			setSerial(null);
			
			setMessage(new Text("MSG_SETCARD"));
			
			thread = new Thread() {
				
				/**
				 * Prepare a field and a method to interrupt this thread because
				 * it is impossible to abort the waitForCard method of OCF.
				 * 
				 * スレッド停止のためのフィールドとメソッド。
				 * OCFのwaitForCardは途中でキャンセルできないため。
				 */
				private boolean interrupted = false;
				public void interrupt() {
					this.interrupted = true;
				}
				
				/**
				 * 1.repeats the waitForCard method invocation with a timeout of 1 sec.
				 * 2.if this thread interrupted(interrupted==null)
				 *  2-1.displays a message.
				 *  2-2.ends the while loop.
				 * 3.if a Card object obtained(card!=null)
				 *  3-1.obtains a CardService object.
				 *  3-2.obtains a mutex lock of the card.
				 *  3-3.obtains serial number.
				 *  3-4.displays serial number.
				 *  3-5.releases the mutex locl of the card.
				 *  3-6.ends the while loop.
				 * 4.sets LED state.
				 * 
				 * 1.タイムアウト1秒で、waitForCardを繰り返す。
				 * 2.中断された場合（interrupted==true）
				 *  2-1.メッセージ表示。
				 *  2-2.ループを抜ける。
				 * 3.カードが、セットされた場合（card!=null)
				 *  3-1.カードサービスを取得する。
				 *  3-2.カードをロックする。
				 *  3-3.シリアル番号を読み込む。
				 *  3-4.シリアル番号を表示する。
				 *  3-5.カードのロックを解除する。
				 *  3-6.ループを抜ける。
				 * 4.LEDを設定する。
				 */
				public void run() {
					
					SmartCard card = null;
					
					try {
						CardRequest cr = new CardRequest(
								CardRequest.ANYCARD, null, null);
						cr.setTimeout(1);
						
						while (true) {
							if (interrupted) {
								setMessage(new Text("ERR_ABORT"));
								break;
							}
							
							card = SmartCard.waitForCard(cr);
							
							if (card != null) {
								FileAccessCardService fileAccessService =
									(FileAccessCardService)card.getCardService(
											FileAccessCardService.class, false);
								
								byte[] serialNo = fileAccessService.read(filePath, 0, fileLen);
								
								setSerial(serialNo);
								
								setMessage(new Text("MSG_SUCCESS"));
								
								break;
							}
						}
						
					} catch (CardTerminalException e) {
						setMessage(new Text("ERR_CARDTERMINAL"));
						e.printStackTrace();
						
					} catch (CardServiceException e) {
						setMessage(new Text("ERR_CARDSERVICE"));
						e.printStackTrace();
						
					} catch (ClassNotFoundException e) {
						setMessage(new Text("ERR_CLASSNOTFOUND"));
						e.printStackTrace();
						
					} catch (Throwable t) {
						setMessage(new Text("ERR_THROWABLE"));
						t.printStackTrace();
						
					} finally {
						if (card != null) {
							
							try {
								card.close();
								
							} catch (CardTerminalException e) {
								setMessage(new Text("ERR_CARDTERMINAL"));
								e.printStackTrace();
							}
						}
						
						setLED(Type.START, Operation.GRN_ON);
					}
				}
			};
			thread.start();
			
			setLED(Type.START, Operation.RED_BLINK);
		}
	}
	
	/**
	 * The process followed after the claer/stop key is pressed.
	 * 1.interrupts the thread waiting for cards.
	 * 
	 * クリア/ストップキーが、押された時の処理
	 * 1.スレッドを停止する。
	 */
	synchronized private void pushClearStopKey() {
		if ((thread != null) && thread.isAlive()) {
			
			thread.interrupt();
			
			try {
				thread.join();
				
			} catch (InterruptedException e) {
			}
		}
	}
	
	/**
	 * Displays the given serial number on the serial number label.
	 * 
	 * シリアル番号を、シリアル番号表示ラベルに表示する。
	 */
	private void setSerial(byte[] serialNo) {
		final String replace = "NUMBER";
		Text text = new Text("SERIAL_LABEL");
		
		if (serialNo == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < serialNo.length; i++) {
				n = serialNo[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		serialLabel.setText(text);
		serialLabel.repaint();
	}
	
	/**
	 * Displays the given message on the message label.
	 * 
	 * メッセージを、メッセージ表示ラベルに表示する。
	 */
	private void setMessage(Text message) {
		msgLabel.setText(message);
		msgLabel.repaint();
	}
	
	/**
	 * Activates LED with the given Type and the given Operation.
	 * 
	 * LEDを動作させる。
	 */
	private void setLED(Type type, Operation ope) {
		try {
			new LED(frame, type, ope).blink();
		} catch (LEDExecutionException e) {
			e.printStackTrace();
		}
	}
}
